from django.contrib import admin
from django.core.mail import send_mail
from django.conf import settings
from django import forms
from django.utils.timezone import now
from django.urls import path
from django.shortcuts import render
from import_export import resources, fields
from import_export.widgets import ForeignKeyWidget
from import_export.admin import ImportExportModelAdmin
from .models import Project, Task, Comment, Attachment, DailyTaskUpdate, ProjectDocument, ProjectFeedback
from django.contrib.auth import get_user_model
from django.contrib.admin import SimpleListFilter
from django.utils.translation import gettext_lazy as _
from datetime import timedelta, date


User = get_user_model()

admin.site.site_header = "Project Management Admin"
admin.site.site_title = "Project Admin Portal"
admin.site.index_title = "Welcome to the Project Management Dashboard"


# --- PROJECT TYPE FILTER ---
class ProjectTypeFilter(admin.SimpleListFilter):
    title = 'Project Type'
    parameter_name = 'project_type'

    def lookups(self, request, model_admin):
        return [
            ('1', 'New Project'),
            ('0', 'Old Project'),
        ]

    def queryset(self, request, queryset):
        if self.value() == '1':
            return queryset.filter(project_type=True)
        if self.value() == '0':
            return queryset.filter(project_type=False)
        return queryset


# --- PROJECT RESOURCE ---
class ProjectResource(resources.ModelResource):
    project_progress_in_name = fields.Field(
        column_name='Project Progress',
        attribute='project_progress_in',
        widget=ForeignKeyWidget(Project._meta.get_field('project_progress_in').related_model, 'name')
    )

    project_type_display = fields.Field(
        column_name='Project Type',
        attribute='project_type',
    )

    class Meta:
        model = Project
        fields = (
            'name', 'status', 'start_date', 'end_date',
            'project_progress_in_name', 'project_type_display','status_details'
            # 'ui_days', 'frontend_days', 'backend_days', 'mobile_app_days',
            # 'testing_days', 'bug_solve_days', 'retest_days',
            # 'final_bug_solve_days', 'production_days',
        )
        export_order = fields

    def dehydrate_project_type_display(self, obj):
        return "New Project" if obj.project_type else "Old Project"



class ProjectTypeFilter(SimpleListFilter):
    title = _('Project Type')
    parameter_name = 'project_type'

    def lookups(self, request, model_admin):
        return (
            ('new', _('New Project')),
            ('old', _('Old Project')),
        )

    def queryset(self, request, queryset):
        if self.value() == 'new':
            return queryset.filter(project_type=True)
        elif self.value() == 'old':
            return queryset.filter(project_type=False)
        return queryset


# ✅ Custom End Date Range Filter
class EndDateRangeFilter(SimpleListFilter):
    title = _('End Date Range')
    parameter_name = 'end_date_range'

    def lookups(self, request, model_admin):
        return (
            ('next_7_days', _('Next 7 Days')),
            ('next_15_days', _('Next 15 Days')),
        )

    def queryset(self, request, queryset):
        today = date.today()
        if self.value() == 'next_7_days':
            return queryset.filter(end_date__gte=today, end_date__lte=today + timedelta(days=7))
        elif self.value() == 'next_15_days':
            return queryset.filter(end_date__gte=today, end_date__lte=today + timedelta(days=15))
        return queryset

class DelayedProjectFilter(SimpleListFilter):
    title = 'Delayed Projects'
    parameter_name = 'is_delayed'

    def lookups(self, request, model_admin):
        return (
            ('yes', 'Delayed'),
            ('no', 'On Time / Completed'),
        )

    def queryset(self, request, queryset):
        if self.value() == 'yes':
            return queryset.filter(end_date__lt=now().date()).exclude(status='completed')
        if self.value() == 'no':
            return queryset.exclude(end_date__lt=now().date()) | queryset.filter(status='completed')
        return queryset

# ✅ Admin for Project model
@admin.register(Project)
class ProjectAdmin(ImportExportModelAdmin):
    resource_class = ProjectResource
    change_list_template = "admin/projects/project/change_list.html"  # ✅ Force Django to use this template

    list_display = (
        'name', 'status', 'start_date', 'end_date', 'project_progress_in',
        'ui_days', 'frontend_days', 'backend_days', 'mobile_app_days',
        'testing_days', 'bug_solve_days', 'retest_days',
        'final_bug_solve_days', 'production_days',
        'get_project_type_display', 'is_delayed'
    )
    search_fields = ('name','end_date')

    def get_project_type_display(self, obj):
        return "New Project" if obj.project_type else "Old Project"
    get_project_type_display.short_description = 'Project Type'

    list_filter = (
        'status',
        'project_progress_in',
        'start_date',
        'end_date',
        EndDateRangeFilter,
        ProjectTypeFilter,
        DelayedProjectFilter, 
    )
    list_filter_position = 'top'

    exclude = [
        'ui_days', 'frontend_days', 'backend_days', 'mobile_app_days',
        'testing_days', 'bug_solve_days', 'retest_days',
        'final_bug_solve_days', 'production_days'
    ]

    def get_urls(self):
        urls = super().get_urls()
        custom_urls = [
            path('delayed-projects/', self.admin_site.admin_view(self.delayed_projects_view), name='delayed-projects'),
        ]
        return custom_urls + urls

    def delayed_projects_view(self, request):
        delayed_projects = Project.objects.filter(end_date__lt=now()).exclude(status='completed')
        context = {
            **self.admin_site.each_context(request),
            'delayed_projects': delayed_projects,
            'title': 'Delayed Projects Report',
        }
        return render(request, 'admin/delayed_projects.html', context)

    def is_delayed(self, obj):
        return obj.end_date < now().date() and obj.status != 'completed'
    is_delayed.boolean = True
    is_delayed.short_description = "Delayed?"

    def changelist_view(self, request, extra_context=None):
        queryset = self.get_queryset(request)

        total_projects = queryset.count()
        in_progress = queryset.filter(status='in_progress').count()
        completed = queryset.filter(status='completed').count()
        old_projects = queryset.filter(project_type=False).count()
        new_projects = queryset.filter(project_type=True).count()
        delayed_projects = queryset.filter(end_date__lt=now(), status__iexact='in_progress').count()
        total_developers = User.objects.filter(groups__name="Team Member").count()

        extra_context = extra_context or {}
        extra_context.update({
            'total_projects': total_projects,
            'in_progress': in_progress,
            'completed': completed,
            'old_projects': old_projects,
            'new_projects': new_projects,
            'delayed_projects': delayed_projects,
            'total_developers': total_developers,
        })

        return super().changelist_view(request, extra_context=extra_context)



# --- DAILY TASK UPDATE FORM ---
class DailyTaskUpdateForm(forms.ModelForm):
    class Meta:
        model = DailyTaskUpdate
        exclude = ('updated_by',)

    def __init__(self, *args, **kwargs):
        self.request = kwargs.pop('request', None)
        super().__init__(*args, **kwargs)

        if self.request:
            user = self.request.user
            if 'task' in self.fields:
                if user.groups.filter(name="Team Member").exists():
                    self.fields['task'].queryset = Task.objects.filter(assigned_to=user)
                elif user.groups.filter(name="Team Leader").exists():
                    self.fields['task'].queryset = Task.objects.filter(assigned_by=user)
                else:
                    self.fields['task'].queryset = Task.objects.all()

            if 'progress_percent' in self.fields:
                if user.groups.filter(name="Team Member").exists():
                    self.fields['progress_percent'].widget = forms.HiddenInput()
                    self.fields['progress_percent'].required = False
                elif not user.is_superuser and not user.groups.filter(name="Team Leader").exists():
                    self.fields['progress_percent'].widget.attrs['readonly'] = True


# --- PROGRESS FILTER ---
class ProgressPercentRangeFilter(admin.SimpleListFilter):
    title = 'Progress Percentage Range'
    parameter_name = 'progress_percent_range'

    def lookups(self, request, model_admin):
        return [
            ('20-50', '20% - 50%'),
            ('50-75', '50% - 75%'),
            ('75-90', '75% - 90%'),
            ('90-100', '90% - 100%')
        ]

    def queryset(self, request, queryset):
        val = self.value()
        ranges = {
            '20-50': (20, 50),
            '50-75': (50, 75),
            '75-90': (75, 90),
            '90-100': (90, 100),
        }
        if val in ranges:
            low, high = ranges[val]
            return queryset.filter(progress_percent__gte=low, progress_percent__lt=high if high != 100 else high)
        return queryset


# --- DAILY TASK UPDATE ADMIN ---
class DailyTaskUpdateResource(resources.ModelResource):
    task = fields.Field(column_name='Task Title', attribute='task',
                        widget=ForeignKeyWidget(Task, 'title'))

    class Meta:
        model = DailyTaskUpdate
        fields = ('task', 'date', 'status', 'progress_percent', 'remarks')
        export_order = fields


# @admin.register(DailyTaskUpdate)
class DailyTaskUpdateAdmin(ImportExportModelAdmin):
    resource_class = DailyTaskUpdateResource
    form = DailyTaskUpdateForm
    list_display = ('task', 'date', 'status', 'progress_percent')
    list_filter = ('status', 'date', ProgressPercentRangeFilter)
    search_fields = ('task__title', 'remarks')
    readonly_fields = ('date',)

    def get_readonly_fields(self, request, obj=None):
        ro = list(super().get_readonly_fields(request, obj))
        if request.user.groups.filter(name="Team Member").exists():
            ro.append('progress_percent')
        return ro

    def get_form(self, request, obj=None, **kwargs):
        Form = super().get_form(request, obj, **kwargs)

        class FormWithRequest(Form):
            def __new__(cls, *args, **kwargs):
                kwargs['request'] = request
                return Form(*args, **kwargs)

        return FormWithRequest


# --- TASK ADMIN ---
class TaskResource(resources.ModelResource):
    project_name = fields.Field(column_name='Project Name', attribute='project',
                                widget=ForeignKeyWidget(Project, 'name'))
    assigned_to_name = fields.Field(column_name='Assigned To', attribute='assigned_to',
                                    widget=ForeignKeyWidget(User, 'first_name'))
    assigned_by_name = fields.Field(column_name='Assigned By', attribute='assigned_by',
                                    widget=ForeignKeyWidget(User, 'first_name'))

    class Meta:
        model = Task
        fields = ('title', 'project_name', 'assigned_to_name', 'assigned_by_name',
                  'status', 'priority', 'due_date', 'description')
        export_order = fields


# @admin.register(Task)
class TaskAdmin(ImportExportModelAdmin):
    resource_class = TaskResource
    list_display = ('title', 'project', 'assigned_to', 'status', 'priority', 'due_date')
    readonly_fields = ('assigned_by',)

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        if request.user.is_superuser:
            return qs
        elif request.user.groups.filter(name="Team Member").exists():
            return qs.filter(assigned_to=request.user)
        elif request.user.groups.filter(name="Team Leader").exists():
            return qs.filter(assigned_by=request.user)
        return qs.none()

    def changelist_view(self, request, extra_context=None):
        qs = self.get_queryset(request)
        extra_context = extra_context or {}
        extra_context.update({
            'total_tasks': qs.count(),
            'completed_tasks': qs.filter(status='done').count(),
            'in_progress_tasks': qs.filter(status='in_progress').count(),
            'pending_tasks': qs.filter(status='pending').count(),
            'blocked_tasks': qs.filter(status='blocked').count(),
        })
        return super().changelist_view(request, extra_context=extra_context)

    def save_model(self, request, obj, form, change):
        if not change:
            obj.assigned_by = request.user
        super().save_model(request, obj, form, change)

        if obj.assigned_to and obj.assigned_to.email:
            subject = f"New Task Assigned: {obj.title}"
            message = f"""
Hi {obj.assigned_to.first_name},

You have been assigned a new task in the project: {obj.project.name}.

Task Details:
-------------
Title       : {obj.title}
Description : {obj.description}
Due Date    : {obj.due_date}
Priority    : {obj.priority.capitalize()}

Please log in to your dashboard to view the task details.

Regards,
Project Management System
"""
            try:
                send_mail(subject, message, settings.DEFAULT_FROM_EMAIL, [obj.assigned_to.email], fail_silently=False)
            except Exception as e:
                print("Error sending mail:", e)


# --- COMMENT ADMIN ---
# @admin.register(Comment)
class CommentAdmin(admin.ModelAdmin):
    list_display = ('task', 'user', 'created_at')


# --- ATTACHMENT ADMIN ---
# @admin.register(Attachment)
class AttachmentAdmin(admin.ModelAdmin):
    list_display = ('task', 'uploaded_by', 'created_at')




@admin.register(ProjectDocument)
class ProjectDocumentAdmin(admin.ModelAdmin):
    list_display = ('project', 'document', 'uploaded_by', 'uploaded_at')
    search_fields = ('project__name',)


from django.contrib import admin
from .models import ProjectFeedback, ProjectFeedbackImage
from django.core.mail import EmailMessage
from django.conf import settings
from django.utils.html import format_html
from django import forms


class ProjectFeedbackImageInline(admin.TabularInline):
    model = ProjectFeedbackImage
    extra = 1


# Optional: Custom form to add date filtering logic (not required yet)
class ProjectFeedbackForm(forms.ModelForm):
    class Meta:
        model = ProjectFeedback
        fields = '__all__'


import os
from uuid import uuid4
from mimetypes import guess_type
from django.contrib import admin
from django.core.mail import EmailMessage
from django.conf import settings
from email.mime.image import MIMEImage

from .models import ProjectFeedback, ProjectFeedbackImage


from django.db import transaction

@admin.register(ProjectFeedback)
class ProjectFeedbackAdmin(admin.ModelAdmin):
    inlines = [ProjectFeedbackImageInline]
    list_display = ['project', 'date', 'created_at']
    list_filter = ['project', 'date']
    filter_horizontal = ['users_to_notify']
    search_fields = ['project__name', 'users_to_notify__email']

    def save_model(self, request, obj, form, change):
        super().save_model(request, obj, form, change)
        transaction.on_commit(lambda: self.send_feedback_email(obj))

    def send_feedback_email(self, instance):
        from django.core.mail import EmailMessage
        from mimetypes import guess_type
        from uuid import uuid4
        from email.mime.image import MIMEImage
        from django.conf import settings
        import os
        from PIL import Image
        from io import BytesIO

        subject = f"Feedback Received from {instance.project.name} on {instance.date}"
        recipient_list = list(instance.users_to_notify.values_list('email', flat=True))

        html_lines = [
            '<div style="font-family: Arial, sans-serif; font-size: 14px;">',
            f'<h2 style="color: #2E86C1;">Feedback Report: {instance.project.name}</h2>',
            f'<p><strong>Date:</strong> {instance.date.strftime("%d %B %Y")}</p>',
            '<p>Hello Team,</p>',
            '<p>Kindly check the feedback received from client side:</p>',
        ]

        attachments = []
        has_content = False

        email = EmailMessage(subject, "", settings.DEFAULT_FROM_EMAIL, recipient_list)
        email.content_subtype = "html"

        images = instance.images.all()

        if not images.exists():
            html_lines.append("<p>No feedback or images provided.</p>")
        else:
            for index, img in enumerate(images, start=1):
                if img.image:
                    image_path = img.image.path
                    original_name = os.path.basename(img.image.name)
                    mime_type, _ = guess_type(image_path)

                    html_lines.append(f"<h3 style='color:#117A65;'>Feedback {index}</h3>")

                    if mime_type and mime_type.startswith("image/"):
                        cid = f'image{uuid4().hex[:8]}'

                        img_pil = Image.open(image_path)
                        img_pil.thumbnail((500, 500))  # Resize to max width/height
                        img_io = BytesIO()
                        img_pil.save(img_io, format=img_pil.format or "JPEG", quality=95)
                        img_io.seek(0)

                        mime_image = MIMEImage(img_io.read(), _subtype=(img_pil.format or "jpeg").lower())
                        mime_image.add_header('Content-ID', f'<{cid}>')
                        mime_image.add_header('Content-Disposition', 'inline', filename=original_name)
                        attachments.append(mime_image)

                        html_lines.append(
                            f'<img src="cid:{cid}" style="max-width:600px; border: 1px solid #ddd; '
                            f'padding: 4px; border-radius: 5px; display:block;" />'
                        )
                        has_content = True
                    else:
                        with open(image_path, 'rb') as f:
                            file_content = f.read()
                            email.attach(original_name, file_content, mime_type or 'application/octet-stream')
                        html_lines.append(f"<p><strong>Attachment:</strong> {original_name}</p>")
                        has_content = True

                if img.content:
                    formatted_html = self.convert_feedback_text_to_html(img.content)
                    html_lines.append(
                        f'<div style="margin-top: 10px; background-color: #f9f9f9; '
                        f'padding: 10px; border-left: 4px solid #2E86C1;">'
                        f'{formatted_html}</div>'
                    )
                    has_content = True

        html_lines.append('</div>')
        email.body = "\n".join(html_lines)

        # Attach embedded images (with Content-ID and filename)
        for attachment in attachments:
            email.attach(attachment)

        email.send(fail_silently=False)

    def convert_feedback_text_to_html(self, content):
        lines = content.strip().split("\n")
        html = ""
        in_list = False

        for line in lines:
            line = line.strip()
            if line.startswith("-"):
                if not in_list:
                    html += "<ul>"
                    in_list = True
                html += f"<li>{line[1:].strip()}</li>"
            else:
                if in_list:
                    html += "</ul>"
                    in_list = False
                if line.startswith("🔨") or line.startswith("🧱") or line.startswith("🛠️") or line.startswith("###"):
                    html += f"<h4>{line}</h4>"
                elif line.startswith("##"):
                    html += f"<h3>{line}</h3>"
                elif line.startswith("#"):
                    html += f"<h2>{line}</h2>"
                else:
                    html += f"<p>{line}</p>"

        if in_list:
            html += "</ul>"
        return html






# import os
# @admin.register(ProjectFeedback)
# class ProjectFeedbackAdmin(admin.ModelAdmin):
#     form = ProjectFeedbackForm
#     inlines = [ProjectFeedbackImageInline]

#     list_display = ['project', 'date', 'created_at']
#     list_filter = ['project', 'date']
#     filter_horizontal = ['users_to_notify']
#     search_fields = ['project__name', 'users_to_notify__email']

#     def save_related(self, request, form, formsets, change):
#         super().save_related(request, form, formsets, change)

#         # ✅ Prevent duplicate email sending
#         if hasattr(form.instance, '_email_sent'):
#             return
#         form.instance._email_sent = True

#         instance = form.instance

#         subject = f"Feedback Received from {instance.project.name} on {instance.date}"
#         recipient_list = list(instance.users_to_notify.values_list('email', flat=True))

#         # ✅ Email body
#         body_lines = ['Hello Team', "", "Kindly check the feedback received from client side"]
#         for img in instance.images.all():
#             if img.content:
#                 body_lines.append(f"- {img.content}")
#         message = "\n".join(body_lines) if body_lines else "No description provided for images."

#         email = EmailMessage(subject, message, settings.DEFAULT_FROM_EMAIL, recipient_list)

#         # ✅ Handle attachments, rename .zip files temporarily
#         renamed_files = []  # Keep track of renamed files to restore later
#         for img in instance.images.all():
#             if img.image:
#                 file_path = img.image.path

#                 if file_path.lower().endswith('.zip'):
#                     safe_path = file_path + ".safe"
#                     os.rename(file_path, safe_path)
#                     email.attach_file(safe_path)
#                     renamed_files.append((safe_path, file_path))  # Store for restoration
#                 else:
#                     email.attach_file(file_path)

#         # ✅ Send the email
#         email.send(fail_silently=False)

#         # ✅ Restore original zip filenames
#         for safe_path, original_path in renamed_files:
#             if os.path.exists(safe_path):
#                 os.rename(safe_path, original_path)


from django.contrib import admin
from .models import TeamProgress

from import_export.admin import ExportMixin


class TeamProgressResource(resources.ModelResource):
    class Meta:
        model = TeamProgress
        fields = (
            'project__name', 'team__name', 'assigned_start_date', 'expected_end_date',
            'actual_end_date', 'rework_start_date', 'rework_end_date',
            'delay_reason', 'delay_days', 'rework_days', 'updated_at'
        )

@admin.register(TeamProgress)
class TeamProgressAdmin(ExportMixin, admin.ModelAdmin):
    resource_class = TeamProgressResource

    list_display = (
        'project', 
        'team', 
        'assigned_start_date', 
        'expected_end_date', 
        'actual_end_date',
        'rework_start_date',
        'rework_end_date',
        'delay_reason',
        'delay_days_display',
        'rework_days_display',
        'updated_at'
    )
    list_filter = (
        'team',
        'delay_reason',
        'assigned_start_date',
        'expected_end_date',
        'actual_end_date',
        'updated_at'
    )
    search_fields = ('project__name', 'team__name', 'remarks')
    date_hierarchy = 'assigned_start_date'
    ordering = ('-updated_at',)

    def delay_days_display(self, obj):
        return obj.delay_days
    delay_days_display.short_description = 'Delay (Days)'

    def rework_days_display(self, obj):
        return obj.rework_days
    rework_days_display.short_description = 'Rework (Days)'








