from django.db import models
from django.contrib.auth import get_user_model
from teams.models import Team
from datetime import timedelta, date
from django.core.exceptions import ValidationError
from math import isclose


User = get_user_model()


class Project(models.Model):
    STATUS_CHOICES = [
        ('planning', 'Planning'),
        ('in_progress', 'In Progress'),
        ('completed', 'Completed'),
        ('on_hold', 'On Hold'),
          ('cancelled', 'Cancelled'),
    ]

    name = models.CharField(max_length=255)
    
    # description = models.TextField(blank=True)
    start_date = models.DateField()
    end_date = models.DateField()
    project_type = models.BooleanField(default=True, help_text="Check if this is a special type of project")
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='planning')
    status_details = models.TextField(null=True, blank=True)
    created_by = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, related_name='created_projects')
    project_progress_in = models.ForeignKey(Team, on_delete=models.SET_NULL, null=True, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)

    # Percentages
    ui_percentage = models.FloatField(default=0)
    frontend_percentage = models.FloatField(default=0)
    backend_percentage = models.FloatField(default=0)
    mobile_app_percentage = models.FloatField(default=0)
    testing_percentage = models.FloatField(default=0)
    bug_solve_percentage = models.FloatField(default=0)
    retest_percentage = models.FloatField(default=0)
    final_bug_solve_percentage = models.FloatField(default=0)
    production_percentage = models.FloatField(default=0)

    # Working days (calculated from percentage)
    ui_days = models.PositiveIntegerField(default=0)
    frontend_days = models.PositiveIntegerField(default=0)
    backend_days = models.PositiveIntegerField(default=0)
    mobile_app_days = models.PositiveIntegerField(default=0)
    testing_days = models.PositiveIntegerField(default=0)
    bug_solve_days = models.PositiveIntegerField(default=0)
    retest_days = models.PositiveIntegerField(default=0)
    final_bug_solve_days = models.PositiveIntegerField(default=0)
    production_days = models.PositiveIntegerField(default=0)

    def total_percentage(self):
        return (
            self.ui_percentage +
            self.frontend_percentage +
            self.backend_percentage +
            self.mobile_app_percentage +
            self.testing_percentage +
            self.bug_solve_percentage +
            self.retest_percentage +
            self.final_bug_solve_percentage +
            self.production_percentage
        )

    def clean(self):
        total = self.total_percentage()

        if not isclose(total, 100.0, abs_tol=0.01):
            raise ValidationError(f"Total percentage must be exactly 100%. Currently: {total:.2f}%.")

        if not self.start_date or not self.end_date:
            raise ValidationError("Start date and end date are required to calculate working days.")

    def calculate_working_days(self):
        if not self.start_date or not self.end_date:
            return 0

        working_days = 0
        current_day = self.start_date
        while current_day <= self.end_date:
            if current_day.weekday() < 5:  # Monday to Friday only
                working_days += 1
            current_day += timedelta(days=1)
        return working_days

    def save(self, *args, **kwargs):
        self.full_clean()  # This calls the clean() method and validates
        total_days = self.calculate_working_days()

        self.ui_days = round((self.ui_percentage / 100.0) * total_days)
        self.frontend_days = round((self.frontend_percentage / 100.0) * total_days)
        self.backend_days = round((self.backend_percentage / 100.0) * total_days)
        self.mobile_app_days = round((self.mobile_app_percentage / 100.0) * total_days)
        self.testing_days = round((self.testing_percentage / 100.0) * total_days)
        self.bug_solve_days = round((self.bug_solve_percentage / 100.0) * total_days)
        self.retest_days = round((self.retest_percentage / 100.0) * total_days)
        self.final_bug_solve_days = round((self.final_bug_solve_percentage / 100.0) * total_days)
        self.production_days = round((self.production_percentage / 100.0) * total_days)

        super().save(*args, **kwargs)


    def __str__(self):
        return self.name
    


class Task(models.Model):
    STATUS_CHOICES = [
        ('pending', 'Pending'),
        ('in_progress', 'In Progress'),
        ('done', 'Done'),
        ('blocked', 'Blocked'),
    ]

    PRIORITY_CHOICES = [
        ('low', 'Low'),
        ('medium', 'Medium'),
        ('high', 'High'),
    ]

    project = models.ForeignKey(Project, on_delete=models.CASCADE)
    title = models.CharField(max_length=255)
    description = models.TextField(blank=True)
    assigned_to = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, related_name='assigned_tasks')
    assigned_by = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, related_name='assigned_by_tasks')
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pending')
    priority = models.CharField(max_length=10, choices=PRIORITY_CHOICES, default='medium')
    start_date = models.DateField()
    due_date = models.DateField()
    completed_at = models.DateField(null=True, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return self.title


class Comment(models.Model):
    task = models.ForeignKey(Task, on_delete=models.CASCADE, related_name='comments')
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    content = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"Comment by {self.user} on {self.task.title}"


class Attachment(models.Model):
    task = models.ForeignKey(Task, on_delete=models.CASCADE, related_name='attachments')
    file = models.FileField(upload_to='attachments/')
    uploaded_by = models.ForeignKey(User, on_delete=models.SET_NULL, null=True)
    created_at = models.DateTimeField(auto_now_add=True)


class DailyTaskUpdate(models.Model):
    STATUS_CHOICES = [
        ('todo', 'To Do'),
        ('in_progress', 'In Progress'),
        ('review', 'In Review'),
        ('completed', 'Completed'),
    ]

    task = models.ForeignKey(Task, on_delete=models.CASCADE, related_name='daily_updates')
    updated_by = models.ForeignKey(User, on_delete=models.CASCADE)
    date = models.DateField(auto_now_add=True)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES)
    remarks = models.TextField()
    progress_percent = models.PositiveIntegerField(default=0)  # 0–100%

    def __str__(self):
        return f"{self.task.title} - {self.status} ({self.progress_percent}%)"

from django.conf import settings

from django.utils.text import slugify
from datetime import date
import os

def project_feedback_upload_to(instance, filename):
    project_name = slugify(instance.feedback.project.name)
    return os.path.join('project_feedback', project_name, filename)

class ProjectDocument(models.Model):
    project = models.ForeignKey(Project, on_delete=models.CASCADE, related_name='documents')
    document = models.FileField(upload_to='project_documents/')
    uploaded_by = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, blank=True)
    uploaded_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.project.name} - {self.document.name}"


class ProjectFeedback(models.Model):
    project = models.ForeignKey('Project', on_delete=models.CASCADE, related_name='feedbacks')
    date = models.DateField(default=date.today)
    
    users_to_notify = models.ManyToManyField(User, related_name='feedback_notifications')

    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"Feedback for {self.project.name} on {self.date}"


class ProjectFeedbackImage(models.Model):
    feedback = models.ForeignKey(ProjectFeedback, on_delete=models.CASCADE, related_name='images')
    image = models.FileField(upload_to=project_feedback_upload_to, null=True, blank=True)
    content = models.TextField(null=True,blank=True)

    def __str__(self):
        return f"Image for feedback {self.feedback.id}"
    

class TeamProgress(models.Model):
    DELAY_REASON_CHOICES = [
        ('none', 'No Delay'),
        ('client', 'Client-side Delay'),
        ('internal', 'Internal/Team Delay'),
        ('other', 'Other Circumstances'),
    ]

    project = models.ForeignKey(Project, on_delete=models.CASCADE, related_name='team_progresses')
    team = models.ForeignKey(Team, on_delete=models.CASCADE, related_name='progress_entries')

    assigned_start_date = models.DateField()
    expected_end_date = models.DateField()
    actual_end_date = models.DateField(null=True, blank=True)

    rework_start_date = models.DateField(null=True, blank=True)
    rework_end_date = models.DateField(null=True, blank=True)

    delay_reason = models.CharField(
        max_length=20,
        choices=DELAY_REASON_CHOICES,
        default='none'
    )

    remarks = models.TextField(null=True, blank=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return f"{self.team.name} - {self.project.name}"
    
    @property
    def delay_days(self):
        if self.actual_end_date and self.expected_end_date:
            delta = (self.actual_end_date - self.expected_end_date).days
            return delta if delta > 0 else 0
        return 0

    @property
    def rework_days(self):
        if self.rework_start_date and self.rework_end_date:
            return (self.rework_end_date - self.rework_start_date).days
        return 0