from django.contrib import admin
from django.urls import reverse
from django.utils.html import format_html
from django.contrib.auth.models import User
from .models import Team, TeamMember

@admin.register(Team)
class TeamAdmin(admin.ModelAdmin):
    list_display = ['name']

@admin.register(TeamMember)
class TeamMemberAdmin(admin.ModelAdmin):
    list_display = [ 'user_first_name','user_last_name','user']
    # list_filter = ['team']
    def user_first_name(self, obj):
        return obj.user.first_name
    user_first_name.short_description = 'First Name'

    def user_last_name(self, obj):
        return obj.user.last_name
    user_last_name.short_description = 'Last Name'
    
    def changelist_view(self, request, extra_context=None):
        from django.db.models import Count

        # Get total developers in group
        total_developers = User.objects.filter(groups__name="Team Member").count()

        # Get team-wise member counts
        team_member_counts = Team.objects.annotate(member_count=Count('teammember'))

        extra_context = extra_context or {}
        extra_context['total_developers'] = total_developers
        extra_context['team_member_counts'] = team_member_counts

        return super().changelist_view(request, extra_context=extra_context)


from django.contrib import admin
from .models import WeeklyMemberRating

from django.contrib import admin
from django.db.models import Sum
from .models import WeeklyMemberRating

from django.db.models import Sum, Subquery, OuterRef

@admin.register(WeeklyMemberRating)
class WeeklyMemberRatingAdmin(admin.ModelAdmin):
    list_display = (
        'member',
        'week_start_date',
        'week_end_date',
        'rating',
        'remarks',
        'created_at',
        'total_user_rating',
    )
    search_fields = ('rating',)
    list_filter = ['rating']
    # ✅ Remove this — it only supports real fields
    # ordering = ['-total_rating']

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.annotate(
            total_rating=Subquery(
                WeeklyMemberRating.objects
                .filter(member=OuterRef('member'))
                .values('member')
                .annotate(total=Sum('rating'))
                .values('total')[:1]
            )
        ).order_by('-total_rating')  # ✅ Add ordering here

    def total_user_rating(self, obj):
        return round(getattr(obj, 'total_rating', 0) or 0, 2)

    total_user_rating.short_description = 'Total Rating'
    total_user_rating.admin_order_field = 'total_rating'  # Allows column sort in UI
